/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <string.h>

#include "_xitk.h"
#include "inputtext.h"
#include "button.h"
#include "slider.h"
#include "font.h"
#include "backend.h"

typedef enum {
  /* keep order */
  _W_input = 0,
  _W_more,
  _W_less,
  _W_slider,
  /* /keep order */
  _W_LAST
} _W_t;

/** intbox *********************************************************/

#include "intbox.h"

typedef struct {
  xitk_widget_t        w;

  xitk_intbox_widget_t info;
  int                  step[5];
  int                  input_width, slider_width, pm_width;
  uint32_t             last_pos;
  xitk_slider_hv_t     hv;
  xitk_widget_t       *iw[_W_LAST];
  uint8_t              buf[32];
} _intbox_private_t;

static const uint8_t _ib_xitk_modes[8] = {
  [INTBOX_FMT_SECONDS]    = 1,
  [INTBOX_FMT_MILLISECS]  = 2
};

static const uint8_t _ib_hex[16] = "0123456789abcdef";

static const char *_ib_set_text (_intbox_private_t *wp) {
  uint8_t *q = wp->buf + sizeof (wp->buf);
  *--q = 0;
  if (wp->info.fmt == INTBOX_FMT_DECIMAL) {
    uint32_t v = wp->info.value < 0 ? -wp->info.value : wp->info.value;
    do {
      *--q = '0' + (v % 10u);
      v /= 10u;
    } while (v);
    if (wp->info.value < 0)
      *--q = '-';
  } else if (wp->info.fmt == INTBOX_FMT_SECONDS) {
    uint32_t v = wp->info.value < 0 ? -wp->info.value : wp->info.value;
    *--q = '0' + (v % 10u);
    v /= 10u;
    *--q = '0' + (v % 6u);
    v /= 6u;
    *--q = ':';
    *--q = '0' + (v % 10u);
    v /= 10u;
    *--q = '0' + (v % 6u);
    v /= 6u;
    *--q = ':';
    do {
      *--q = '0' + (v % 10u);
      v /= 10u;
    } while (v);
    if (wp->info.value < 0)
      *--q = '-';
  } else if (wp->info.fmt == INTBOX_FMT_MILLISECS) {
    uint32_t v = wp->info.value < 0 ? -wp->info.value : wp->info.value;
    *--q = '0' + (v % 10u);
    v /= 10u;
    *--q = '0' + (v % 10u);
    v /= 10u;
    *--q = '0' + (v % 10u);
    v /= 10u;
    *--q = '.';
    do {
      *--q = '0' + (v % 10u);
      v /= 10u;
    } while (v);
    if (wp->info.value < 0)
      *--q = '-';
  } else {
    uint32_t v = wp->info.value;
    do {
      *--q = _ib_hex[v & 0x0f];
      v >>= 4;
    } while (v);
    if (wp->info.fmt == INTBOX_FMT_0x) {
      *--q = 'x';
      *--q = '0';
    } else if (wp->info.fmt == INTBOX_FMT_HASH) {
      *--q = '#';
    }
  }
  if (wp->iw[_W_input] && strcmp ((const char *)q, xitk_inputtext_get_text (wp->iw[_W_input])))
    xitk_inputtext_change_text (wp->iw[_W_input], (const char *)q);
  return (const char *)q;
}

static void _ib_set_slider (_intbox_private_t *wp) {
  if (!wp->iw[_W_slider])
    return;
  wp->hv.h.pos = wp->info.value - wp->info.min;
  wp->hv.h.step = 1;
  wp->hv.h.visible = 1;
  wp->hv.h.max = wp->info.max - wp->info.min + 1;
  wp->hv.v.pos = 0;
  wp->hv.v.step = 1;
  wp->hv.v.visible = 0;
  wp->hv.v.max = 0;
  xitk_slider_hv_sync (wp->iw[_W_slider], &wp->hv,
    (wp->w.state & XITK_WIDGET_STATE_VISIBLE) ? XITK_SLIDER_SYNC_SET_AND_PAINT : XITK_SLIDER_SYNC_SET);
}

static _intbox_private_t *_ib_ptr (xitk_widget_t *w) {
  _intbox_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INTBOX)) ? wp : NULL;
}

static int _ib_set_value (_intbox_private_t *wp, int value) {
  if (value > wp->info.max)
    value = wp->info.max;
  else if (value < wp->info.min)
    value = wp->info.min;
  if (value != wp->info.value) {
    wp->info.value = value;
    _ib_set_text (wp);
    _ib_set_slider (wp);
  }
  return value;
}

static int _ib_event (xitk_widget_t *w, const widget_event_t *event) {
  XITK_HV_INIT;
  _intbox_private_t *wp = _ib_ptr (w);

  if (!wp)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_KEY:
      {
        uint32_t u = 0;
        int v;
        if (event->string[0] == XITK_CTRL_KEY_PREFIX) {
          static const uint8_t key_map[XITK_KEY_LASTCODE] = {
            [XITK_KEY_UP]           = 3 * 2,
            [XITK_KEY_DOWN]         = 2 * 2,
            [XITK_KEY_PREV]         = 4 * 2,
            [XITK_KEY_NEXT]         = 1 * 2,
            [XITK_MOUSE_WHEEL_UP]   = 3 * 2 + 1,
            [XITK_MOUSE_WHEEL_DOWN] = 2 * 2 + 1
          };
          const uint8_t *s = (const uint8_t *)event->string;
          u = (s[1] >= XITK_KEY_LASTCODE) ? 0 : key_map[s[1]];
        } else if (event->string[0] == '-') {
          u = 2 * 2;
        } else if (event->string[0] == '+') {
          u = 3 * 2;
        }
        if (!u)
          break;
        if (!event->pressed)
          return 1;
        v = wp->info.value;
        if (v != _ib_set_value (wp, wp->info.value + wp->step[u >> 1] * ((u & 1) * event->button + 1))) {
          if (wp->info.callback)
            wp->info.callback (&wp->w, wp->info.nw.userdata, wp->info.value, event->modifier);
        }
        return 1;
      }
    case WIDGET_EVENT_PAINT:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        if (wp->w.pos.w != wp->last_pos) {
          wp->last_pos = wp->w.pos.w;
          xitk_set_widget_pos (wp->iw[_W_input], XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
          xitk_set_widget_pos (wp->iw[_W_more], XITK_HV_H (wp->w.pos) + wp->input_width, XITK_HV_V (wp->w.pos));
          xitk_set_widget_pos (wp->iw[_W_less],
            XITK_HV_H (wp->w.pos) + wp->input_width, XITK_HV_V (wp->w.pos) + XITK_HV_V (wp->w.size)- wp->pm_width);
          xitk_set_widget_pos (wp->iw[_W_slider], XITK_HV_H (wp->w.pos) + wp->input_width + 2, XITK_HV_V (wp->w.pos));
        }
      }
      xitk_widgets_state (wp->iw + _W_input, 4, XITK_WIDGET_STATE_VISIBLE, wp->w.state);
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      break;
    case WIDGET_EVENT_DESTROY:
      xitk_widgets_delete (wp->iw + _W_input, 4);
      break;
    case WIDGET_EVENT_ENABLE:
      xitk_widgets_state (wp->iw + _W_input, 4, XITK_WIDGET_STATE_ENABLE, (wp->w.state & XITK_WIDGET_STATE_ENABLE));
      break;
    case WIDGET_EVENT_TIPS_TIMEOUT:
      xitk_set_widget_tips_and_timeout (wp->iw[_W_input], wp->w.tips_string, event->tips_timeout);
      xitk_set_widget_tips_and_timeout (wp->iw[_W_slider], wp->w.tips_string, event->tips_timeout);
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static void _ib_it (xitk_widget_t *x, void *data, const char *string) {
  _intbox_private_t *wp = (_intbox_private_t *)data;
  int v;

  if (!string)
    return;

  (void)x;
  v = xitk_str2int32 (&string, _ib_xitk_modes[wp->info.fmt & 7]);
  if (v > wp->info.max)
    v = wp->info.max;
  else if (v < wp->info.min)
    v = wp->info.min;
  if (v != wp->info.value) {
    wp->info.value = v;
    _ib_set_text (wp);
    _ib_set_slider (wp);
    if (wp->info.callback)
      wp->info.callback (&wp->w, wp->info.nw.userdata, wp->info.value, 0);
  } else {
    _ib_set_text (wp);
  }
}

/*
 *
 */
int xitk_intbox_set_value (xitk_widget_t *w, int value) {
  _intbox_private_t *wp = _ib_ptr (w);

  return wp ? _ib_set_value (wp, value) : 0;
}

/*
 *
 */
int xitk_intbox_get_value (xitk_widget_t *w) {
  _intbox_private_t *wp = _ib_ptr (w);

  return wp ? wp->info.value : 0;
}

/*
 *
 */
static void _ib_step (xitk_widget_t *x, void *data) {
  _intbox_private_t *wp = (_intbox_private_t *)data;
  int v;

  if (x == wp->iw[_W_more]) {
    v = wp->info.value + wp->info.step;
    if (v > wp->info.max)
      v = wp->info.max;
  } else {
    v = wp->info.value - wp->info.step;
    if (v < wp->info.min)
      v = wp->info.min;
  }
  if (v != wp->info.value) {
    wp->info.value = v;
    _ib_set_text (wp);
    _ib_set_slider (wp);
    if (wp->info.callback)
      wp->info.callback (&wp->w, wp->info.nw.userdata, wp->info.value, 0);
  }
}

static void _ib_sl (xitk_widget_t *x, void *data, int pos, unsigned int modifier) {
  _intbox_private_t *wp = (_intbox_private_t *)data;
  int v;

  (void)x;
  (void)pos;
  xitk_slider_hv_sync (wp->iw[_W_slider], &wp->hv, XITK_SLIDER_SYNC_GET);
  v = wp->info.min + wp->hv.h.pos;
  if (v != wp->info.value) {
    wp->info.value = v;
    _ib_set_text (wp);
    if (wp->info.callback)
      wp->info.callback (&wp->w, wp->info.nw.userdata, wp->info.value, modifier);
  }
}

/*
 *
 */
xitk_widget_t *xitk_noskin_intbox_create (const xitk_intbox_widget_t *ib, int x, int y, int width, int height) {
  _intbox_private_t *wp;
  char fontname[128];
  XITK_HV_INIT;

  wp = (_intbox_private_t *)xitk_widget_new (&ib->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_H (wp->w.size) = width;
  XITK_HV_V (wp->w.size) = height;
  wp->w.state   &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.type     = WIDGET_GROUP | WIDGET_TYPE_INTBOX | WIDGET_KEYABLE;
  wp->w.event    = _ib_event;

  wp->info = *ib;
  wp->step[0] = 0;
  wp->step[2] = -wp->info.step;
  wp->step[3] = wp->info.step;
  if (wp->info.fmt == INTBOX_FMT_MILLISECS) {
    wp->step[1] = -1000;
    wp->step[4] = 1000;
  } else {
    wp->step[1] = -wp->info.step * 10;
    wp->step[4] = wp->info.step * 10;
  }

  if (wp->info.min >= wp->info.max) {
    int v = ~(unsigned int)0 >> 1;
    wp->info.max = v;
    wp->info.min = -v - 1;
  }
  if (wp->info.value > wp->info.max)
    wp->info.value = wp->info.max;
  else if (wp->info.value < wp->info.min)
    wp->info.value = wp->info.min;

  /* Create inputtext and buttons (not skinable) */
  if (XITK_HV_H (wp->w.size) >= 7 * XITK_HV_V (wp->w.size)) {
    wp->input_width = (5 * XITK_HV_V (wp->w.size)) >> 1;
    wp->slider_width = XITK_HV_H (wp->w.size) - wp->input_width - 2;
    wp->pm_width = 0;
  } else {
    wp->pm_width = XITK_HV_V (wp->w.size) >> 1;
    wp->input_width = XITK_HV_H (wp->w.size) - wp->pm_width;
    wp->slider_width = 0;
  }

  wp->iw[_W_input] = NULL;
  {
    xitk_inputtext_widget_t inp = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX
      },
      .text = _ib_set_text (wp),
      .max_length = 32,
      .callback = _ib_it
    };

    XITK_DEFAULT_FONT_NAME (fontname, DEFAULT_FONT_FMT, height);
    wp->iw[_W_input] = xitk_noskin_inputtext_create (&inp,
      x, y, wp->input_width, XITK_HV_V (wp->w.size), XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);
  }

  if (wp->pm_width) {
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_TABABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX
      },
      .callback = _ib_step,
      .symbol = XITK_SYMBOL_PLUS
    };

    wp->iw[_W_more] = xitk_noskin_button_create (&b,
      x + wp->input_width, y, wp->pm_width, wp->pm_width);

    b.symbol = XITK_SYMBOL_MINUS;
    wp->iw[_W_less] = xitk_noskin_button_create (&b,
      x + wp->input_width, y + XITK_HV_V (wp->w.size) - wp->pm_width, wp->pm_width, wp->pm_width);
  }

  if (wp->slider_width) {
    xitk_slider_widget_t sl = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_INTBOX
      },
      .step = 1,
      .type = XITK_HVSLIDER,
      .callback = _ib_sl,
      .motion_callback = _ib_sl
    };

    wp->iw[_W_slider] = xitk_noskin_slider_create (&sl,
        x + wp->input_width + 2, y, wp->slider_width, XITK_HV_V (wp->w.size));
    _ib_set_slider (wp);
  }

  xitk_widget_set_focus_redirect (&wp->w, wp->iw[_W_input]);

  return _xitk_new_widget_apply (&ib->nw, &wp->w);
}

/** doublebox ******************************************************/

#include "doublebox.h"

typedef struct {
  xitk_widget_t          w;

  xitk_widget_t         *iw[_W_LAST];

  double                 step;
  double                 value;

  xitk_double_callback_t callback;

  uint32_t               last_pos;

  char                   buf[80];
} _doublebox_private_t;

static void _db_set (_doublebox_private_t *wp, int more) {
  wp->buf[0] = 0;
  snprintf (wp->buf, sizeof (wp->buf), "%lf", wp->value);
  if (more > 0)
    xitk_inputtext_change_text (wp->iw[_W_input], wp->buf);
  if (more > 1)
    if (wp->callback)
      wp->callback (&wp->w, wp->w.userdata, wp->value);
}

static int _db_event (xitk_widget_t *w, const widget_event_t *event) {
  XITK_HV_INIT;
  _doublebox_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_KEY:
      if (event->string[0] == XITK_CTRL_KEY_PREFIX) {
        static const int8_t key_map[XITK_KEY_LASTCODE] = {
          [XITK_KEY_UP]           =   1 * 2,
          [XITK_KEY_DOWN]         =  -1 * 2,
          [XITK_KEY_PREV]         =  10 * 2,
          [XITK_KEY_NEXT]         = -10 * 2,
          [XITK_MOUSE_WHEEL_UP]   =   1 * 2 + 1,
          [XITK_MOUSE_WHEEL_DOWN] =  -1 * 2 + 1
        };
        const uint8_t *s = (const uint8_t *)event->string;
        int v = (s[1] >= XITK_KEY_LASTCODE) ? 0 : key_map[s[1]];
        if (!v)
          break;
        if (event->pressed) {
          wp->value += wp->step * ((v >> 1) * ((v & 1) * event->button + 1));
          _db_set (wp, 2);
        }
      }
      return 1;
    case WIDGET_EVENT_PAINT:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        if (wp->w.pos.w != wp->last_pos) {
          int bx, ih, iw;
          wp->last_pos = wp->w.pos.w;
          iw = xitk_get_widget_width (wp->iw[_W_input]);
          ih = xitk_get_widget_height (wp->iw[_W_input]);
          xitk_set_widget_pos (wp->iw[_W_input], XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
          bx = XITK_HV_H (wp->w.pos) + iw;
          xitk_set_widget_pos (wp->iw[_W_more], bx, XITK_HV_V (wp->w.pos));
          xitk_set_widget_pos (wp->iw[_W_less], bx, XITK_HV_V (wp->w.pos) + (ih >> 1));
        }
      }
      xitk_widgets_state (wp->iw + _W_input, 3, XITK_WIDGET_STATE_VISIBLE, wp->w.state);
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      break;
    case WIDGET_EVENT_DESTROY:
      xitk_widgets_delete (wp->iw + _W_input, 3);
      break;
    case WIDGET_EVENT_ENABLE:
      xitk_widgets_state (wp->iw + _W_input, 3, XITK_WIDGET_STATE_ENABLE, (wp->w.state & XITK_WIDGET_STATE_ENABLE));
      break;
    case WIDGET_EVENT_TIPS_TIMEOUT:
      if (wp->iw[_W_input])
        xitk_set_widget_tips_and_timeout (wp->iw[_W_input], wp->w.tips_string, event->tips_timeout);
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static void _doublebox_it (xitk_widget_t *x, void *data, const char *string) {
  _doublebox_private_t *wp = (_doublebox_private_t *)data;

  (void)x;
  wp->value = strtod (string, NULL);
  _db_set (wp, 2);
}

/*
 *
 */
void xitk_doublebox_set_value(xitk_widget_t *w, double value) {
  _doublebox_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return;

  wp->value = value;
  _db_set (wp, 1);
}

/*
 *
 */
double xitk_doublebox_get_value(xitk_widget_t *w) {
  _doublebox_private_t *wp;
  const char *strval;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return 0;

  strval = xitk_inputtext_get_text (wp->iw[_W_input]);
  wp->value = strtod (strval, NULL);
  return wp->value;
}

/*
 *
 */
static void _doublebox_step (xitk_widget_t *x, void *data) {
  _doublebox_private_t *wp = (_doublebox_private_t *)data;
  double d = (x == wp->iw[_W_more]) ? wp->step : -wp->step;

  wp->value += d;
  _db_set (wp, 2);
}

/*
 *
 */
xitk_widget_t *xitk_noskin_doublebox_create (const xitk_doublebox_widget_t *ib, int x, int y, int width, int height) {
  _doublebox_private_t *wp;
  char fontname[128];
  XITK_HV_INIT;

  wp = (_doublebox_private_t *)xitk_widget_new (&ib->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_H (wp->w.size) = width;
  XITK_HV_V (wp->w.size) = height;
  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.type   = WIDGET_GROUP | WIDGET_TYPE_DOUBLEBOX | WIDGET_KEYABLE;
  wp->w.event  = _db_event;

  wp->callback = ib->callback;
  wp->step     = ib->step;
  wp->value    = ib->value;
  /* Create inputtext and buttons (not skinable) */

  _db_set (wp, 0);
  {
    xitk_inputtext_widget_t inp = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX
      },
      .text = wp->buf,
      .max_length = 16,
      .callback = _doublebox_it
    };

    XITK_DEFAULT_FONT_NAME (fontname, DEFAULT_FONT_FMT, height);
    wp->iw[_W_input] = xitk_noskin_inputtext_create (&inp,
      x, y, (width - 10), height, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);
  }

  {
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX
      },
      .symbol = XITK_SYMBOL_PLUS,
      .callback = _doublebox_step
    };

    wp->iw[_W_more] = xitk_noskin_button_create (&b, x + width - (height >> 1), y, height >> 1, height >> 1);

    b.symbol = XITK_SYMBOL_MINUS;
    wp->iw[_W_less] = xitk_noskin_button_create (&b, x + width - (height >> 1), y + (height >> 1), height >> 1, height >> 1);
  }

  wp->iw[_W_slider] = NULL;

  return _xitk_new_widget_apply (&ib->nw, &wp->w);
}
